<?php
// --- Database Setup ---
require_once 'db_config.php';

try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS categories (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL UNIQUE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    $pdo->exec("
        CREATE TABLE IF NOT EXISTS prompts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            category_id INT NOT NULL,
            title VARCHAR(255) NOT NULL,
            content TEXT NOT NULL,
            short_code VARCHAR(255) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
        )
    ");
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// --- Handle AJAX Requests ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['create_category'])) {
            $name = trim($_POST['name']);
            if (!empty($name)) {
                $stmt = $pdo->prepare("INSERT INTO categories (name) VALUES (?)");
                $stmt->execute([$name]);
            }
        } elseif (isset($_POST['update_category'])) {
            $stmt = $pdo->prepare("UPDATE categories SET name = ? WHERE id = ?");
            $stmt->execute([$_POST['name'], $_POST['id']]);
        } elseif (isset($_POST['delete_category'])) {
            $stmt = $pdo->prepare("DELETE FROM categories WHERE id = ?");
            $stmt->execute([$_POST['id']]);
        } elseif (isset($_POST['create_prompt'])) {
            $short_code = "[" . $_POST['title'] . "]";
            $stmt = $pdo->prepare("INSERT INTO prompts (category_id, title, content, short_code) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_POST['category_id'], $_POST['title'], $_POST['content'], $short_code]);
        } elseif (isset($_POST['update_prompt'])) {
            $short_code = "[" . $_POST['title'] . "]";
            $stmt = $pdo->prepare("UPDATE prompts SET title = ?, content = ?, short_code = ?, category_id = ? WHERE id = ?");
            $stmt->execute([$_POST['title'], $_POST['content'], $short_code, $_POST['category_id'], $_POST['id']]);
        } elseif (isset($_POST['delete_prompt'])) {
            $stmt = $pdo->prepare("DELETE FROM prompts WHERE id = ?");
            $stmt->execute([$_POST['id']]);
        }
    } catch (PDOException $e) {
        die("Database error: " . $e->getMessage());
    }
    exit;
}

// Fetch data
$categories = $pdo->query("SELECT * FROM categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
$promptsByCategory = [];
foreach ($categories as $category) {
    $stmt = $pdo->prepare("SELECT * FROM prompts WHERE category_id = ?");
    $stmt->execute([$category['id']]);
    $promptsByCategory[$category['id']] = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Prompt Manager</title>
    <link rel="stylesheet" href="style-side.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
</head>
<body>
    <div class="sidebar">
        <h2><i class="fas fa-tasks"></i> Manage Prompts</h2>

        <button onclick="openManageModal()" style="width: 100%; padding: 10px; margin-bottom: 20px; background-color: #28a745; border: none; color: white; cursor: pointer;">
            <i class="fas fa-plus"></i> Manage Category-Prompt
        </button>

        <div class="category-list">
            <h3><i class="fas fa-list"></i> Categories</h3>
            <div id="category-accordion">
                <?php foreach ($categories as $category): ?>
                    <div class="category-item">
                        <div class="category-header" onclick="toggleCategory(<?= $category['id'] ?>)">
                            <?= htmlspecialchars($category['name']) ?>
                            <div class="action-icons">
                                <i class="fas fa-edit" onclick="openEditCategoryModal(event, <?= $category['id'] ?>, '<?= htmlspecialchars($category['name'], ENT_QUOTES) ?>')"></i>
                                <i class="fas fa-trash" onclick="confirmDeleteCategory(event, <?= $category['id'] ?>)"></i>
                            </div>
                        </div>
                        <div class="category-content" id="content-<?= $category['id'] ?>">
                            <?php foreach ($promptsByCategory[$category['id']] as $prompt): ?>
                                    <div class="prompt-item"
     onclick='insertPromptIntoChat(<?= json_encode([
         'short_code' => $prompt['short_code'],
         'content' => $prompt['content']
     ], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT) ?>)'>
                                    <?= htmlspecialchars($prompt['title']) ?>
                                    <div class="action-icons">
                                        <i class="fas fa-edit" onclick="openEditPromptModal(event, <?= $prompt['id'] ?>, '<?= htmlspecialchars($prompt['title'], ENT_QUOTES) ?>', `<?= str_replace('`', '\`', htmlspecialchars($prompt['content'], ENT_QUOTES)) ?>`, <?= $prompt['category_id'] ?>)"></i>
                                        <i class="fas fa-trash" onclick="confirmDeletePrompt(event, <?= $prompt['id'] ?>)"></i>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <!-- Manage Modal -->
    <div id="manageModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal('manageModal')">&times;</span>
            <h2><i class="fas fa-tasks"></i> Manage Category-Prompt</h2>

            <div class="category-section">
                <h3><i class="fas fa-plus-circle"></i> Create Category</h3>
                <form method="POST" id="category-form">
                    <input type="text" name="name" id="category-name" placeholder="Category Name" required>
                    <button type="submit" class="btn" name="create_category"><i class="fas fa-save"></i> Save</button>
                </form>
            </div>

            <div class="prompt-section">
                <h3><i class="fas fa-plus-circle"></i> Create Prompt</h3>
                <form method="POST" id="prompt-form">
                    <input type="text" name="title" id="prompt-title" placeholder="Prompt Title" required>
                    <textarea name="content" id="prompt-content" placeholder="Prompt Content" required></textarea>
                    <select name="category_id" id="prompt-category" required>
                        <option value="">Select Category</option>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?= htmlspecialchars($category['id']) ?>"><?= htmlspecialchars($category['name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="btn" name="create_prompt"><i class="fas fa-save"></i> Create Prompt</button>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Category Modal -->
    <div id="editCategoryModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal('editCategoryModal')">&times;</span>
            <h2><i class="fas fa-edit"></i> Edit Category</h2>
            <form id="edit-category-form" method="POST">
                <input type="hidden" name="id" id="edit-category-id">
                <input type="text" name="name" id="edit-category-name" required>
                <button type="submit" class="btn" name="update_category"><i class="fas fa-save"></i> Update Category</button>
            </form>
        </div>
    </div>

    <!-- Edit Prompt Modal -->
    <div id="editPromptModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal('editPromptModal')">&times;</span>
            <h2><i class="fas fa-edit"></i> Edit Prompt</h2>
            <form id="edit-prompt-form" method="POST">
                <input type="hidden" name="id" id="edit-prompt-id">
                <input type="text" name="title" id="edit-prompt-title" required>
                <textarea name="content" id="edit-prompt-content" required></textarea>
                <select name="category_id" id="edit-prompt-category" required>
                    <option value="">Select Category</option>
                    <?php foreach ($categories as $category): ?>
                        <option value="<?= htmlspecialchars($category['id']) ?>"><?= htmlspecialchars($category['name']) ?></option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="btn" name="update_prompt"><i class="fas fa-save"></i> Update Prompt</button>
            </form>
        </div>
    </div>

    <!-- Confirmation Modal -->
    <div id="confirmationModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="closeModal('confirmationModal')">&times;</span>
            <h2><i class="fas fa-exclamation-triangle"></i> Confirm Deletion</h2>
            <p id="confirmation-message">Are you sure you want to delete this item?</p>
            <form id="confirmation-form" method="POST">
                <input type="hidden" name="id" id="item-id">
                <input type="hidden" name="item_type" id="item-type">
                <button type="submit" class="delete-btn" name="confirm_delete"><i class="fas fa-check"></i> Yes, Delete</button>
                <button type="button" class="edit-btn" onclick="closeModal('confirmationModal')"><i class="fas fa-times"></i> Cancel</button>
            </form>
        </div>
    </div>

    <script>
        // Toggle category visibility
        function toggleCategory(categoryId) {
            const content = document.getElementById(`content-${categoryId}`);
            content.style.display = content.style.display === 'block' ? 'none' : 'block';
        }

        // Insert prompt content into chat input
function insertPromptIntoChat(promptData) {
    let insertText = '';

    // Use only the short_code part
    if (promptData.short_code) {
        if (typeof promptData.short_code === 'object') {
            insertText = JSON.stringify(promptData.short_code, null, 2); // Pretty print
        } else {
            insertText = String(promptData.short_code);
        }
    }

    const input = document.getElementById('user-input');
    if (!input) return;

    const currentValue = input.value;
    const selectionStart = input.selectionStart;
    const selectionEnd = input.selectionEnd;

    input.value = currentValue.substring(0, selectionStart) +
                  insertText +
                  currentValue.substring(selectionEnd);

    input.focus();
    input.setSelectionRange(selectionStart + insertText.length, selectionStart + insertText.length);
}

        // Modal management functions
        function openManageModal() {
            closeAllModals();
            document.getElementById('manageModal').style.display = 'block';
        }

        function openEditCategoryModal(event, categoryId, categoryName) {
            event.stopPropagation();
            closeAllModals();
            const modal = document.getElementById('editCategoryModal');
            modal.style.display = 'block';
            document.getElementById('edit-category-id').value = categoryId;
            document.getElementById('edit-category-name').value = categoryName;
        }

        function openEditPromptModal(event, promptId, promptTitle, promptContent, categoryId) {
            event.stopPropagation();
            closeAllModals();
            const modal = document.getElementById('editPromptModal');
            modal.style.display = 'block';
            document.getElementById('edit-prompt-id').value = promptId;
            document.getElementById('edit-prompt-title').value = promptTitle;
            document.getElementById('edit-prompt-content').value = promptContent;
            document.getElementById('edit-prompt-category').value = categoryId;
        }

        function confirmDeleteCategory(event, categoryId) {
            event.stopPropagation();
            closeAllModals();
            const modal = document.getElementById('confirmationModal');
            modal.style.display = 'block';
            document.getElementById('confirmation-message').textContent = 'Are you sure you want to delete this category and all its prompts?';
            document.getElementById('item-id').value = categoryId;
            document.getElementById('item-type').value = 'category';
        }

        function confirmDeletePrompt(event, promptId) {
            event.stopPropagation();
            closeAllModals();
            const modal = document.getElementById('confirmationModal');
            modal.style.display = 'block';
            document.getElementById('confirmation-message').textContent = 'Are you sure you want to delete this prompt?';
            document.getElementById('item-id').value = promptId;
            document.getElementById('item-type').value = 'prompt';
        }

        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }

        function closeAllModals() {
            document.querySelectorAll('.modal').forEach(modal => {
                modal.style.display = 'none';
            });
        }

        // Form submission handlers
        document.addEventListener('DOMContentLoaded', function() {
            // Category form
            document.getElementById('category-form').addEventListener('submit', function(e) {
                e.preventDefault();
                submitForm(this, 'create_category');
            });

            // Prompt form
            document.getElementById('prompt-form').addEventListener('submit', function(e) {
                e.preventDefault();
                submitForm(this, 'create_prompt');
            });

            // Edit category form
            document.getElementById('edit-category-form').addEventListener('submit', function(e) {
                e.preventDefault();
                submitForm(this, 'update_category');
            });

            // Edit prompt form
            document.getElementById('edit-prompt-form').addEventListener('submit', function(e) {
                e.preventDefault();
                submitForm(this, 'update_prompt');
            });

            // Confirmation form
            document.getElementById('confirmation-form').addEventListener('submit', function(e) {
                e.preventDefault();
                const formData = new FormData();
                const itemType = document.getElementById('item-type').value;
                const itemId = document.getElementById('item-id').value;

                if (itemType === 'category') {
                    formData.append('delete_category', '1');
                    formData.append('id', itemId);
                } else if (itemType === 'prompt') {
                    formData.append('delete_prompt', '1');
                    formData.append('id', itemId);
                }

                fetch('', {
                    method: 'POST',
                    body: formData
                }).then(res => {
                    if (res.ok) location.reload();
                });
            });
        });

        // Generic form submission function
        function submitForm(form, action) {
            const formData = new FormData(form);
            formData.append(action, '1');

            fetch('', {
                method: 'POST',
                body: formData
            }).then(res => {
                if (res.ok) location.reload();
            });
        }

        // Close modal when clicking outside of it
        window.addEventListener('click', function(event) {
            if (event.target.classList.contains('modal')) {
                closeAllModals();
            }
        });
    </script>
    
    
    
    
    
    
    
    
</body>
</html>
