// Initialize PDF.js
pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdn.jsdelivr.net/npm/pdfjs-dist@3.4.120/build/pdf.worker.min.js';

// DOM Elements
const apiKeyInput = document.getElementById('api-key');
const saveApiKeyBtn = document.getElementById('save-api-key-btn');
const uploadBtn = document.getElementById('upload-btn');
const fileInput = document.getElementById('pdf-upload');
const fileNameDisplay = document.getElementById('file-name');
const pdfPreview = document.getElementById('pdf-preview');
const processBtn = document.getElementById('process-btn');
const downloadBtn = document.getElementById('download-btn');
const resultsSection = document.getElementById('results-section');
const htmlPreview = document.getElementById('html-preview');
const imagesContent = document.getElementById('images-content');
const loadingIndicator = document.getElementById('loading');
const tabButtons = document.querySelectorAll('.tab-btn');
const tabPanes = document.querySelectorAll('.tab-pane');
const dropArea = document.getElementById('drop-area');

// State
let pdfFile = null;
let currentPdfUrl = null;
let ocrResult = null;

// Load saved API key from local storage
const savedApiKey = localStorage.getItem('mistralApiKey');
if (savedApiKey) {
    apiKeyInput.value = savedApiKey;
}

// PDF Preview Functions
async function loadPdfPreview(pdfUrl) {
    pdfPreview.innerHTML = '';

    try {
        const loadingTask = pdfjsLib.getDocument(pdfUrl);
        const pdf = await loadingTask.promise;

        const pagesContainer = document.createElement('div');
        pagesContainer.className = 'pdf-pages';
        pdfPreview.appendChild(pagesContainer);

        const pageControls = document.createElement('div');
        pageControls.className = 'page-controls';
        pageControls.innerHTML = `
            <div class="page-info">Page 1 of ${pdf.numPages}</div>
            <div class="page-nav">
                <button id="prev-page" disabled>Previous</button>
                <button id="next-page" ${pdf.numPages === 1 ? 'disabled' : ''}>Next</button>
            </div>
        `;
        pdfPreview.appendChild(pageControls);

        let currentPage = 1;
        await renderPage(pdf, currentPage, pagesContainer);

        document.getElementById('prev-page').addEventListener('click', async () => {
            if (currentPage > 1) {
                currentPage--;
                pagesContainer.innerHTML = '';
                await renderPage(pdf, currentPage, pagesContainer);
                updatePageControls(currentPage, pdf.numPages);
            }
        });

        document.getElementById('next-page').addEventListener('click', async () => {
            if (currentPage < pdf.numPages) {
                currentPage++;
                pagesContainer.innerHTML = '';
                await renderPage(pdf, currentPage, pagesContainer);
                updatePageControls(currentPage, pdf.numPages);
            }
        });
    } catch (error) {
        console.error('Error loading PDF preview:', error);
        pdfPreview.innerHTML = '<div class="error">Failed to load PDF preview</div>';
    }
}

async function renderPage(pdf, pageNum, container) {
    const page = await pdf.getPage(pageNum);
    const viewport = page.getViewport({ scale: 1.0 });
    const containerWidth = container.clientWidth || pdfPreview.clientWidth;
    const scale = containerWidth / viewport.width;
    const scaledViewport = page.getViewport({ scale: scale * 0.95 });

    const canvas = document.createElement('canvas');
    const context = canvas.getContext('2d');
    canvas.height = scaledViewport.height;
    canvas.width = scaledViewport.width;

    const renderContext = {
        canvasContext: context,
        viewport: scaledViewport
    };

    await page.render(renderContext).promise;
    container.appendChild(canvas);
}

function updatePageControls(currentPage, totalPages) {
    const pageInfo = document.querySelector('.page-info');
    const prevButton = document.getElementById('prev-page');
    const nextButton = document.getElementById('next-page');

    pageInfo.textContent = `Page ${currentPage} of ${totalPages}`;
    prevButton.disabled = currentPage === 1;
    nextButton.disabled = currentPage === totalPages;
}

// Drag and Drop Handlers
function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
}

function highlight() {
    dropArea.classList.add('active');
}

function unhighlight() {
    dropArea.classList.remove('active');
}

function handleDrop(e) {
    const dt = e.dataTransfer;
    const files = dt.files;

    if (files.length > 0 && files[0].type === 'application/pdf') {
        handleFiles(files);
    } else {
        alert('Please drop a PDF file');
    }
}

function handleFiles(files) {
    if (files.length > 0) {
        pdfFile = files[0];
        fileNameDisplay.textContent = `Selected file: ${pdfFile.name}`;

        if (currentPdfUrl) {
            URL.revokeObjectURL(currentPdfUrl);
        }
        currentPdfUrl = URL.createObjectURL(pdfFile);
        loadPdfPreview(currentPdfUrl);
        processBtn.disabled = false;
    }
}

// OCR Processing Functions
async function uploadPdfToMistral(apiKey, pdfFile) {
    const formData = new FormData();
    formData.append('purpose', 'ocr');
    formData.append('file', pdfFile);

    const uploadResponse = await fetch('https://api.mistral.ai/v1/files', {
        method: 'POST',
        headers: {
            'Authorization': `Bearer ${apiKey}`
        },
        body: formData
    });

    if (!uploadResponse.ok) {
        let errorMessage;
        try {
            const errorData = await uploadResponse.json();
            errorMessage = errorData.error?.message || uploadResponse.statusText;
        } catch (e) {
            errorMessage = `Status ${uploadResponse.status}: ${uploadResponse.statusText}`;
        }
        throw new Error(`File upload failed: ${errorMessage}`);
    }

    return await uploadResponse.json();
}

async function processPdfWithOcr(apiKey, pdfFile) {
    try {
        const fileData = await uploadPdfToMistral(apiKey, pdfFile);
        const fileId = fileData.id;

        const urlResponse = await fetch(`https://api.mistral.ai/v1/files/${fileId}/url?expiry=24`, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            }
        });

        if (!urlResponse.ok) {
            throw new Error(`Failed to get signed URL: ${urlResponse.status}`);
        }

        const urlData = await urlResponse.json();
        const signedUrl = urlData.url;

        const ocrResponse = await fetch('https://api.mistral.ai/v1/ocr', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            },
            body: JSON.stringify({
                model: "mistral-ocr-latest",
                document: {
                    type: "document_url",
                    document_url: signedUrl
                },
                include_image_base64: true
            })
        });

        if (!ocrResponse.ok) {
            let errorDetail;
            try {
                const errorData = await ocrResponse.json();
                errorDetail = JSON.stringify(errorData);
            } catch (e) {
                errorDetail = `Status ${ocrResponse.status}: ${ocrResponse.statusText}`;
            }
            throw new Error(`OCR processing failed: ${errorDetail}`);
        }

        return await ocrResponse.json();
    } catch (error) {
        console.error('Error in OCR process:', error);
        throw error;
    }
}

function displayOcrResults(result) {
    let htmlText = '';

    if (!result) {
        htmlPreview.innerHTML = '<p>No data received from the API</p>';
        return;
    }

    if (result.text) {
        htmlText = result.text;
    } else if (result.content) {
        htmlText = result.content;
    } else if (result.pages && Array.isArray(result.pages)) {
        htmlText = result.pages
            .filter(page => page !== undefined)
            .map(page => {
                if (!page) return '';
                return page.text || page.content || '';
            })
            .join('\n\n');
    }

    if (htmlText) {
        htmlPreview.innerHTML = htmlText;
    } else {
        htmlPreview.innerHTML = '<p>No text content available in the response</p>';
    }

    imagesContent.innerHTML = '';
    if (result.images && Array.isArray(result.images) && result.images.length > 0) {
        result.images.forEach((image, index) => {
            if (image.base64) {
                const imageCard = document.createElement('div');
                imageCard.className = 'image-card';

                const img = document.createElement('img');
                img.src = `data:image/${image.format || 'png'};base64,${image.base64}`;
                img.alt = `Image ${index + 1}`;

                const caption = document.createElement('div');
                caption.className = 'image-caption';
                caption.textContent = image.caption || `Image ${index + 1}`;

                imageCard.appendChild(img);
                imageCard.appendChild(caption);
                imagesContent.appendChild(imageCard);
            }
        });
    } else if (result.pages && Array.isArray(result.pages)) {
        result.pages.forEach((page, pageIndex) => {
            if (page && page.images && Array.isArray(page.images)) {
                page.images.forEach((image, imageIndex) => {
                    if (image.base64) {
                        const imageCard = document.createElement('div');
                        imageCard.className = 'image-card';

                        const img = document.createElement('img');
                        img.src = `data:image/${image.format || 'png'};base64,${image.base64}`;
                        img.alt = `Page ${pageIndex + 1}, Image ${imageIndex + 1}`;

                        const caption = document.createElement('div');
                        caption.className = 'image-caption';
                        caption.textContent = image.caption || `Page ${pageIndex + 1}, Image ${imageIndex + 1}`;

                        imageCard.appendChild(img);
                        imageCard.appendChild(caption);
                        imagesContent.appendChild(imageCard);
                    }
                });
            }
        });
    }
}

function downloadHtmlResults() {
    const htmlContent = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>OCR Results</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; }
                h1, h2, h3 { margin-top: 1rem; margin-bottom: 0.5rem; }
                p { margin-bottom: 1rem; white-space: pre-line; }
                ul, ol { margin-left: 1.5rem; margin-bottom: 1rem; }
                code { background-color: #f8f9fa; padding: 0.2rem 0.4rem; border-radius: 3px; font-family: monospace; }
                table { border-collapse: collapse; margin-bottom: 1rem; width: 100%; }
                th, td { border: 1px solid #dee2e6; padding: 0.5rem; text-align: left; }
                blockquote { border-left: 3px solid #6c757d; padding-left: 1rem; margin-left: 0; color: #6c757d; white-space: pre-line; }
                img { max-width: 100%; height: auto; margin: 1rem 0; border: 1px solid #dee2e6; border-radius: 4px; }
            </style>
        </head>
        <body>
            <div>${htmlPreview.innerHTML}</div>
            <div>${imagesContent.innerHTML}</div>
        </body>
        </html>
    `;

    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'ocr-results.html';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

function saveApiKey() {
    const apiKey = apiKeyInput.value.trim();
    if (apiKey) {
        localStorage.setItem('mistralApiKey', apiKey);
        alert('API Key saved successfully!');
    } else {
        alert('Please enter an API key before saving.');
    }
}

// Event Listeners Setup
uploadBtn.addEventListener('click', () => {
    fileInput.click();
});

fileInput.addEventListener('change', (e) => {
    handleFiles(e.target.files);
});

processBtn.addEventListener('click', async () => {
    const apiKey = apiKeyInput.value.trim();

    if (!apiKey) {
        alert('Please enter your Mistral AI API key');
        apiKeyInput.focus();
        return;
    }

    if (!pdfFile) {
        alert('Please upload a PDF file first');
        return;
    }
    if (apiKey.length < 20) {
        alert('Please enter a valid Mistral AI API key (typically a long string)');
        apiKeyInput.focus();
        return;
    }

    loadingIndicator.style.display = 'flex';

    try {
        ocrResult = await processPdfWithOcr(apiKey, pdfFile);
        displayOcrResults(ocrResult);
        resultsSection.style.display = 'block';
        resultsSection.scrollIntoView({ behavior: 'smooth' });
        downloadBtn.disabled = false;
    } catch (error) {
        console.error('Error processing PDF:', error);
        if (error.message.includes('401') || error.message.includes('Unauthorized')) {
            alert('Authentication failed: Please check your API key and try again.');
        } else if (error.message.includes('422') || error.message.includes('Unprocessable')) {
            alert('API Error: The request was rejected. This might be due to incompatible parameters or model availability.');
        } else {
            alert(`Error processing PDF: ${error.message}`);
        }
    } finally {
        loadingIndicator.style.display = 'none';
    }
});

downloadBtn.addEventListener('click', () => {
    exportToDocx();
});

saveApiKeyBtn.addEventListener('click', () => {
    saveApiKey();
});

tabButtons.forEach(button => {
    button.addEventListener('click', () => {
        tabButtons.forEach(btn => btn.classList.remove('active'));
        tabPanes.forEach(pane => pane.classList.remove('active'));
        button.classList.add('active');
        const tabId = button.dataset.tab;
        document.getElementById(`${tabId}-content`).classList.add('active');
    });
});

['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
    dropArea.addEventListener(eventName, preventDefaults, false);
});

['dragenter', 'dragover'].forEach(eventName => {
    dropArea.addEventListener(eventName, highlight, false);
});

['dragleave', 'drop'].forEach(eventName => {
    dropArea.addEventListener(eventName, unhighlight, false);
});

dropArea.addEventListener('drop', handleDrop, false);

async function exportToDocx() {
    const tempContainer = document.createElement('div');
    tempContainer.innerHTML = htmlPreview.innerHTML.replace(/<h1>/g, '<h2>').replace(/<\/h1>/g, '</h2>');

    const maxWidth = 624;

    const loadImage = src => new Promise(resolve => {
      const img = new Image();
      img.onload = () => resolve(img);
      img.onerror = () => resolve(null);
      img.src = src;
    });

    const images = tempContainer.querySelectorAll('img');
    for (const img of images) {
      const loadedImg = await loadImage(img.src);
      if (!loadedImg) continue;

      const originalWidth = loadedImg.naturalWidth;
      const originalHeight = loadedImg.naturalHeight;

      if (originalWidth > maxWidth) {
        const ratio = maxWidth / originalWidth;
        img.width = maxWidth;
        img.height = originalHeight * ratio;
      } else {
        img.width = originalWidth;
        img.height = originalHeight;
      }

      img.setAttribute('style', `width:${img.width}px;height:${img.height}px;display:block;margin:10px auto;`);
    }

    tempContainer.querySelectorAll('table').forEach((table) => {
      table.setAttribute('border', '1');
      table.style.borderCollapse = 'collapse';
      table.style.width = '100%';
      table.style.marginBottom = '10px';

      table.querySelectorAll('th, td').forEach((cell) => {
        cell.style.border = '1px solid black';
        cell.style.padding = '5px';
        cell.style.textAlign = 'left';
      });
    });

    const cleanedHTML = tempContainer.innerHTML.replace(/\[\^?\d+\]/g, '');

    const fullHTML = `
      <!DOCTYPE html>
      <html>
      <head>
        <meta charset="utf-8">
        <style>
          body { font-family: Arial, sans-serif; font-size: 12pt; }
          img { display: block; margin: 10px auto; }
          table { border-collapse: collapse; width: 100%; margin-bottom: 10px; }
          th, td { border: 1px solid black; padding: 5px; text-align: left; }
        </style>
      </head>
      <body>${cleanedHTML}</body>
      </html>
    `;

    const blob = htmlDocx.asBlob(fullHTML);
    const originalFilename = pdfFile.name;
    let docxFilename;

    if (/\.pdf$/i.test(originalFilename)) {
        docxFilename = originalFilename.replace(/\.pdf$/i, '.docx');
    } else {
        docxFilename = originalFilename + '.docx';
    }

    saveAs(blob, docxFilename);
}
