<?php
require 'config.php';

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
        header("Location: login.php");
    //echo "Access denied.";
    exit;
}

// Initialize message variable
$msg = '';

// Create user_categories table if it doesn't exist
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS user_categories (
            user_id INT NOT NULL,
            category_id INT NOT NULL,
            PRIMARY KEY (user_id, category_id),
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
        )
    ");
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Add columns to users table if they don't exist
try {
    $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS enabled_until DATETIME DEFAULT NULL");
    $pdo->exec("ALTER TABLE users ADD COLUMN IF NOT EXISTS is_enabled TINYINT(1) DEFAULT 1");
} catch (PDOException $e) {
    // Columns might already exist, continue
}

// Handle account enable/disable
if (isset($_POST['toggle_enable']) && isset($_POST['user_id']) && isset($_POST['enable_action'])) {
    $userId = intval($_POST['user_id']);
    $action = $_POST['enable_action'];

    if ($userId !== $_SESSION['user']['id'] && $userId > 0) {
        try {
            if ($action === 'disable') {
                $stmt = $pdo->prepare("UPDATE users SET is_enabled = 0, enabled_until = NULL WHERE id = ?");
                $stmt->execute([$userId]);
                $msg = "Account disabled successfully.";
            } elseif ($action === 'enable_permanent') {
                $stmt = $pdo->prepare("UPDATE users SET is_enabled = 1, enabled_until = NULL WHERE id = ?");
                $stmt->execute([$userId]);
                $msg = "Account enabled permanently.";
            } else {
                $enableUntil = null;
                switch ($action) {
                    
                    case '2min': $enableUntil = date('Y-m-d H:i:s', strtotime('+1 minutes')); break;
                    case '7days': $enableUntil = date('Y-m-d H:i:s', strtotime('+7 days')); break;
                    case '30days': $enableUntil = date('Y-m-d H:i:s', strtotime('+30 days')); break;
                    case '60days': $enableUntil = date('Y-m-d H:i:s', strtotime('+60 days')); break;
                     case '90days': $enableUntil = date('Y-m-d H:i:s', strtotime('+90 days')); break;
                    case '1year': $enableUntil = date('Y-m-d H:i:s', strtotime('+1 year')); break;
                }

                if ($enableUntil) {
                    $stmt = $pdo->prepare("UPDATE users SET is_enabled = 1, enabled_until = ? WHERE id = ?");
                    $stmt->execute([$enableUntil, $userId]);
                    $msg = "Account enabled until " . $enableUntil . ".";
                }
            }
        } catch (Exception $e) {
            $msg = "Error updating account status: " . $e->getMessage();
        }
    }
}

// Handle user creation
if (isset($_POST['create_user'])) {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $role = $_POST['role'];
    $categories = isset($_POST['categories']) ? $_POST['categories'] : [];

    if ($username && $password && in_array($role, ['admin', 'user'])) {
        // Check if username exists
        $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            $msg = "Username already exists.";
        } else {
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $pdo->beginTransaction();
            try {
                // Insert user
                $stmtInsert = $pdo->prepare("INSERT INTO users (username, password, role, is_enabled, enabled_until) VALUES (?, ?, ?, 1, NULL)");
                $stmtInsert->execute([$username, $hashedPassword, $role]);
                $userId = $pdo->lastInsertId();

                // Insert categories if user role
                if ($role === 'user' && !empty($categories)) {
                    $stmt = $pdo->prepare("INSERT INTO user_categories (user_id, category_id) VALUES (?, ?)");
                    foreach ($categories as $categoryId) {
                        if (intval($categoryId) > 0) {
                            $stmt->execute([$userId, intval($categoryId)]);
                        }
                    }
                }

                $pdo->commit();
                $msg = "User '{$username}' created successfully.";
            } catch (Exception $e) {
                $pdo->rollBack();
                $msg = "Error creating user: " . $e->getMessage();
            }
        }
    } else {
        $msg = "Please fill all required fields correctly.";
    }
}

// Handle user deletion
if (isset($_GET['delete']) && intval($_GET['delete']) > 0) {
    $id = intval($_GET['delete']);
    if ($id !== $_SESSION['user']['id']) {
        $pdo->beginTransaction();
        try {
            // Delete user categories first
            $pdo->prepare("DELETE FROM user_categories WHERE user_id = ?")->execute([$id]);
            // Delete user
            $pdo->prepare("DELETE FROM users WHERE id = ?")->execute([$id]);
            $pdo->commit();
            $msg = "User deleted successfully.";
        } catch (Exception $e) {
            $pdo->rollBack();
            $msg = "Error deleting user: " . $e->getMessage();
        }
    }
}

// Handle user update - OPTIMIZED VERSION
if (isset($_POST['update_user']) && isset($_POST['update_user_id'])) {
    $id = intval($_POST['update_user_id']);
    $username = trim($_POST['update_username']);
    $password = isset($_POST['update_password']) ? trim($_POST['update_password']) : '';
    $role = $_POST['update_role'];
    
    // Get categories array
    $categories = [];
    if (isset($_POST['update_categories']) && is_array($_POST['update_categories'])) {
        $categories = array_map('intval', $_POST['update_categories']);
        $categories = array_filter($categories, function($cat) { return $cat > 0; });
    }

    // Validation
    if (empty($username) || !in_array($role, ['admin', 'user']) || $id <= 0) {
        $msg = "Invalid data provided for user update.";
    } else {
        // Check if username exists for other users
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? AND id != ?");
        $stmt->execute([$username, $id]);
        if ($stmt->fetch()) {
            $msg = "Username '{$username}' already exists for another user.";
        } else {
            $pdo->beginTransaction();
            try {
                // Update user - handle password conditionally
                if (!empty($password)) {
                    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("UPDATE users SET username = ?, password = ?, role = ? WHERE id = ?");
                    $result = $stmt->execute([$username, $hashedPassword, $role, $id]);
                    $passwordChanged = true;
                } else {
                    $stmt = $pdo->prepare("UPDATE users SET username = ?, role = ? WHERE id = ?");
                    $result = $stmt->execute([$username, $role, $id]);
                    $passwordChanged = false;
                }

                if (!$result) {
                    throw new Exception("Failed to update user information");
                }

                // Always delete existing categories first
                $stmt = $pdo->prepare("DELETE FROM user_categories WHERE user_id = ?");
                $stmt->execute([$id]);

                // Insert new categories only for user role
                $categoriesCount = 0;
                if ($role === 'user' && !empty($categories)) {
                    $stmt = $pdo->prepare("INSERT INTO user_categories (user_id, category_id) VALUES (?, ?)");
                    foreach ($categories as $categoryId) {
                        $stmt->execute([$id, $categoryId]);
                        $categoriesCount++;
                    }
                }

                $pdo->commit();
                
                // Success message
                $msg = "User '{$username}' updated successfully";
                if ($passwordChanged) {
                    $msg .= " (password changed)";
                }
                if ($role === 'user') {
                    $msg .= " with {$categoriesCount} categories assigned";
                }
                $msg .= ".";
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $msg = "Error updating user: " . $e->getMessage();
            }
        }
    }
}

// Fetch all users with fresh data
$users = $pdo->query("SELECT * FROM users ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);

// Fetch all categories
$categories = $pdo->query("SELECT * FROM categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

// Helper functions
function getUserCategories($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT category_id FROM user_categories WHERE user_id = ?");
    $stmt->execute([$userId]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
}

function isAccountEnabled($user) {
    if ($user['role'] === 'admin') return true;
    if (!$user['is_enabled']) return false;
    if ($user['enabled_until'] === null) return true;

    $now = new DateTime();
    $enabledUntil = new DateTime($user['enabled_until']);
    return $now < $enabledUntil;
}

function getEnableStatusText($user) {
    if ($user['role'] === 'admin') return 'Active (Admin)';
    if (!$user['is_enabled']) return 'Disabled';
    if ($user['enabled_until'] === null) return 'Enabled (Permanent)';

    $now = new DateTime();
    $enabledUntil = new DateTime($user['enabled_until']);
    if ($now >= $enabledUntil) return 'Disabled (Auto-disabled)';
    return 'Enabled until ' . $enabledUntil->format('Y-m-d H:i:s');
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Admin Panel</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
<style>
    * { box-sizing: border-box; }
    body { 
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
        background: #f8f9fa; 
        margin: 0; 
        padding: 20px; 
    }
    .container { 
        max-width: 1400px; 
        margin: 0 auto; 
        background: white; 
        padding: 30px; 
        border-radius: 10px; 
        box-shadow: 0 2px 10px rgba(0,0,0,0.1); 
    }
    
    h2, h3 { color: #333; text-align: center; margin-bottom: 25px; }
    
    /* Message Styles */
    .message { 
        padding: 15px; 
        margin: 20px 0; 
        border-radius: 5px; 
        font-weight: bold;
        border: 1px solid;
    }
    .message.success { 
        background: #d4edda; 
        color: #155724; 
        border-color: #c3e6cb; 
    }
    .message.error { 
        background: #f8d7da; 
        color: #721c24; 
        border-color: #f5c6cb; 
    }
    
    /* Form Styles */
    .create-form { 
        background: #f8f9fa; 
        padding: 25px; 
        border-radius: 8px; 
        margin-bottom: 30px; 
    }
    .form-row { 
        display: flex; 
        gap: 15px; 
        align-items: end; 
        flex-wrap: wrap; 
        margin-bottom: 15px; 
    }
    .form-group { 
        display: flex; 
        flex-direction: column; 
    }
    .form-group label { 
        margin-bottom: 5px; 
        font-weight: bold; 
        color: #555; 
    }
    .form-group input, .form-group select { 
        padding: 10px; 
        border: 1px solid #ddd; 
        border-radius: 5px; 
        font-size: 14px; 
    }
    .btn { 
        padding: 10px 20px; 
        border: none; 
        border-radius: 5px; 
        cursor: pointer; 
        font-size: 14px; 
        display: inline-flex; 
        align-items: center; 
        gap: 8px; 
    }
    .btn-primary { background: #007bff; color: white; }
    .btn-primary:hover { background: #0056b3; }
    .btn-success { background: #28a745; color: white; }
    .btn-success:hover { background: #1e7e34; }
    .btn-danger { background: #dc3545; color: white; }
    .btn-danger:hover { background: #c82333; }
    .btn-sm { padding: 5px 10px; font-size: 12px; }
    
    /* Category Section */
    .category-section { 
        background: white; 
        border: 1px solid #ddd; 
        border-radius: 5px; 
        padding: 15px; 
        margin-top: 15px; 
    }
    .category-section h4 { margin: 0 0 15px 0; color: #555; }
    .category-grid { 
        display: grid; 
        grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); 
        gap: 10px; 
    }
    .category-item { 
        display: flex; 
        align-items: center; 
        gap: 8px; 
        padding: 5px; 
    }
    .category-item input[type="checkbox"] { 
        margin: 0; 
    }
    
    /* Table Styles */
    .users-table { 
        width: 100%; 
        border-collapse: collapse; 
        margin-top: 25px; 
        font-size: 13px; 
    }
    .users-table th, .users-table td { 
        border: 1px solid #ddd; 
        padding: 12px 8px; 
        text-align: left; 
        vertical-align: top; 
    }
    .users-table th { 
        background: #f8f9fa; 
        font-weight: bold; 
        color: #555; 
    }
    .users-table tr:nth-child(even) { 
        background: #f9f9f9; 
    }
    
    /* Table Form Elements */
    .table-input { 
        width: 100%; 
        padding: 6px; 
        border: 1px solid #ddd; 
        border-radius: 3px; 
        font-size: 12px; 
    }
    .table-input.short { max-width: 100px; }
    .table-input.medium { max-width: 120px; }
    
    /* Status Styles */
    .status-active { color: #28a745; font-weight: bold; }
    .status-disabled { color: #dc3545; font-weight: bold; }
    
    /* Action Controls */
    .action-controls { display: flex; flex-direction: column; gap: 8px; }
    .enable-controls { display: flex; gap: 5px; align-items: center; flex-wrap: wrap; }
    .enable-select { padding: 4px; font-size: 11px; border: 1px solid #ddd; border-radius: 3px; }
    
    /* Category Display */
    .category-display { max-height: 100px; overflow-y: auto; }
    .category-display.inline { 
        display: grid; 
        grid-template-columns: 1fr; 
        gap: 5px; 
    }
    .category-display label { 
        display: flex; 
        align-items: center; 
        gap: 5px; 
        font-size: 11px; 
        margin: 0; 
    }
    
    /* Links */
    .delete-link { color: #dc3545; text-decoration: none; font-size: 11px; }
    .delete-link:hover { text-decoration: underline; }
    
    /* Navigation */
    .nav-links { 
        text-align: center; 
        margin-top: 30px; 
        padding-top: 20px; 
        border-top: 1px solid #eee; 
    }
    .nav-links a { 
        color: #007bff; 
        text-decoration: none; 
        margin: 0 15px; 
        display: inline-flex; 
        align-items: center; 
        gap: 5px; 
    }
    .nav-links a:hover { text-decoration: underline; }
    
    /* Responsive */
    @media (max-width: 768px) {
        .form-row { flex-direction: column; }
        .users-table { font-size: 11px; }
        .container { padding: 15px; }
    }
</style>
</head>
<body>
<div class="container">
    <h2><i class="fa-solid fa-user-shield"></i> Admin Panel</h2>

    <?php if (!empty($msg)): ?>
        <div class="message <?= (strpos($msg, 'Error') !== false || strpos($msg, 'already exists') !== false) ? 'error' : 'success' ?>">
            <?= htmlspecialchars($msg) ?>
        </div>
    <?php endif; ?>

    <!-- Create User Section -->
    <div class="create-form">
        <h3>Create New User</h3>
        <form method="POST">
            <div class="form-row">
                <div class="form-group">
                    <label for="new-username">Username</label>
                    <input type="text" id="new-username" name="username" required>
                </div>
                <div class="form-group">
                    <label for="new-password">Password</label>
                    <input type="password" id="new-password" name="password" required>
                </div>
                <div class="form-group">
                    <label for="new-role">Role</label>
                    <select id="new-role" name="role" required onchange="toggleCreateCategories()">
                        <option value="user">User</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>&nbsp;</label>
                    <button type="submit" name="create_user" class="btn btn-primary">
                        <i class="fa fa-user-plus"></i> Create User
                    </button>
                </div>
            </div>

            <div id="create-categories" class="category-section" style="display: none;">
                <h4>Select Categories for User</h4>
                <div class="category-grid">
                    <?php foreach ($categories as $category): ?>
                        <div class="category-item">
                            <input type="checkbox" name="categories[]" value="<?= $category['id'] ?>" 
                                   id="create-cat-<?= $category['id'] ?>">
                            <label for="create-cat-<?= $category['id'] ?>"><?= htmlspecialchars($category['name']) ?></label>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </form>
    </div>

    <!-- Users Management -->
    <h3>Manage Users</h3>
    <?php if (count($users) > 0): ?>
    <table class="users-table">
        <thead>
            <tr>
                <th style="width: 50px;">ID</th>
                <th style="width: 150px;">Username</th>
                <th style="width: 100px;">Role</th>
                <th style="width: 150px;">Status</th>
                <th style="width: 200px;">Account Control</th>
                <th style="width: 200px;">Categories</th>
                <th style="width: 200px;">Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($users as $user):
                $userCategories = getUserCategories($pdo, $user['id']);
                $isEnabled = isAccountEnabled($user);
                $statusText = getEnableStatusText($user);
            ?>
                <tr>
                    <td><?= $user['id'] ?></td>
                    
                    <!-- Update Form for Each User -->
                    <form method="POST" id="update-form-<?= $user['id'] ?>">
                        <input type="hidden" name="update_user_id" value="<?= $user['id'] ?>">
                        
                        <td>
                            <input type="text" name="update_username" value="<?= htmlspecialchars($user['username']) ?>" 
                                   class="table-input medium" required>
                        </td>
                        
                        <td>
                            <select name="update_role" class="table-input" required 
                                    onchange="toggleUserCategories(<?= $user['id'] ?>)">
                                <option value="user" <?= $user['role'] == 'user' ? 'selected' : '' ?>>User</option>
                                <option value="admin" <?= $user['role'] == 'admin' ? 'selected' : '' ?>>Admin</option>
                            </select>
                        </td>
                    </form>
                    
                    <td>
                        <span class="<?= $isEnabled ? 'status-active' : 'status-disabled' ?>">
                            <?= htmlspecialchars($statusText) ?>
                        </span>
                    </td>
                    
                    <!-- Account Control -->
                    <td>
                        <?php if ($user['id'] != $_SESSION['user']['id']): ?>
                            <div class="enable-controls">
                                <?php if ($isEnabled): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <input type="hidden" name="enable_action" value="disable">
                                        <button type="submit" name="toggle_enable" class="btn btn-danger btn-sm">
                                            <i class="fa fa-ban"></i> Disable
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <select name="enable_action" class="enable-select" required>
                                            <option value="">Select Duration</option>
                                            <option value="2min">2 Minutes</option>
                                            <option value="7days">7 Days</option>
                                            <option value="30days">30 Days</option>
                                            
                                          <option value="60days">60 Days</option>
                                          
                                     <option value="90days">90 Days</option>
                                          
                                            <option value="1year">1 Year</option>
                                            <option value="enable_permanent">Permanent</option>
                                        </select>
                                        <button type="submit" name="toggle_enable" class="btn btn-success btn-sm">
                                            <i class="fa fa-check"></i> Enable
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <em style="color: #666;">Current User</em>
                        <?php endif; ?>
                    </td>
                    
                    <!-- Categories -->
                    <td>
                        <div id="user-categories-<?= $user['id'] ?>" class="category-display inline" 
                             style="<?= $user['role'] === 'admin' ? 'display:none;' : '' ?>">
                            <?php foreach ($categories as $category): ?>
                                <label>
                                    <input type="checkbox" name="update_categories[]" value="<?= $category['id'] ?>" 
                                           form="update-form-<?= $user['id'] ?>"
                                           <?= in_array($category['id'], $userCategories) ? 'checked' : '' ?>>
                                    <?= htmlspecialchars($category['name']) ?>
                                </label>
                            <?php endforeach; ?>
                        </div>
                        <?php if ($user['role'] === 'admin'): ?>
                            <em style="color: #666;">All categories (Admin)</em>
                        <?php elseif (empty($userCategories)): ?>
                            <em style="color: #999;">No categories assigned</em>
                        <?php endif; ?>
                    </td>
                    
                    <!-- Actions -->
                    <td>
                        <div class="action-controls">
                            <input type="password" name="update_password" placeholder="New Password" 
                                   class="table-input short" form="update-form-<?= $user['id'] ?>"
                                   autocomplete="new-password">
                            
                            <button type="submit" name="update_user" form="update-form-<?= $user['id'] ?>" 
                                    class="btn btn-primary btn-sm" 
                                    onclick="return confirmUpdate(<?= $user['id'] ?>, '<?= htmlspecialchars($user['username']) ?>')">
                                <i class="fa fa-save"></i> Update
                            </button>
                            
                            <?php if ($user['id'] != $_SESSION['user']['id']): ?>
                                <a href="?delete=<?= $user['id'] ?>" 
                                   onclick="return confirm('Are you sure you want to delete user: <?= htmlspecialchars($user['username']) ?>?')" 
                                   class="delete-link">
                                    <i class="fa fa-trash"></i> Delete User
                                </a>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    <?php else: ?>
        <p style="text-align: center; color: #666; padding: 20px;">No users found in the system.</p>
    <?php endif; ?>

    <div class="nav-links">
        <a href="dashboard.php"><i class="fa fa-home"></i> Back to Dashboard</a>
        <a href="logout.php"><i class="fa fa-sign-out-alt"></i> Logout</a>
    </div>
</div>

<script>
function toggleCreateCategories() {
    const roleSelect = document.getElementById('new-role');
    const categorySection = document.getElementById('create-categories');
    categorySection.style.display = roleSelect.value === 'user' ? 'block' : 'none';
}

function toggleUserCategories(userId) {
    const form = document.getElementById('update-form-' + userId);
    const roleSelect = form.querySelector('select[name="update_role"]');
    const categoryDiv = document.getElementById('user-categories-' + userId);
    
    if (categoryDiv) {
        categoryDiv.style.display = roleSelect.value === 'user' ? 'block' : 'none';
        
        // If changing to admin, uncheck all categories
        if (roleSelect.value === 'admin') {
            const checkboxes = categoryDiv.querySelectorAll('input[type="checkbox"]');
            checkboxes.forEach(checkbox => checkbox.checked = false);
        }
    }
}

function confirmUpdate(userId, username) {
    const form = document.getElementById('update-form-' + userId);
    const newUsername = form.querySelector('input[name="update_username"]').value;
    const role = form.querySelector('select[name="update_role"]').value;
    const password = form.querySelector('input[name="update_password"]').value;
    
    let message = `Update user "${newUsername}" with role "${role}"`;
    
    if (password.trim() !== '') {
        message += ' and change password';
    }
    
    if (role === 'user') {
        const checkedCategories = form.querySelectorAll('input[name="update_categories[]"]:checked');
        message += ` (${checkedCategories.length} categories selected)`;
    }
    
    message += '?';
    
    return confirm(message);
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    // Initialize create form
    toggleCreateCategories();
    
    // Initialize all user category displays
    <?php foreach ($users as $user): ?>
        toggleUserCategories(<?= $user['id'] ?>);
    <?php endforeach; ?>
    
    // Form validation
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function(e) {
            const usernameField = form.querySelector('input[name="username"], input[name="update_username"]');
            if (usernameField && usernameField.value.trim() === '') {
                e.preventDefault();
                alert('Username is required!');
                usernameField.focus();
                return false;
            }
        });
    });
});
</script>
</body>
</html>