<?php
require 'config.php';

if (isset($_SESSION['user'])) {
    header("Location: dashboard.php");
    exit;
}

// Function to check if account is currently enabled
function isAccountEnabled($user) {
    if (!isset($user['is_enabled']) || !$user['is_enabled']) {
        return false;
    }

    if ($user['enabled_until'] === null) {
        return true; // Permanently enabled
    }

    // Check if temporary enable period has expired
    $now = new DateTime();
    $enabledUntil = new DateTime($user['enabled_until']);

    return $now < $enabledUntil;
}

// Function to get enable message
function getEnableMessage($user) {
    if ($user['enabled_until'] === null) {
        return "Your account has been permanently disabled. Please contact the administrator.";
    }

    $enabledUntil = new DateTime($user['enabled_until']);
    return "Your account is temporarily disabled until " . $enabledUntil->format('Y-m-d H:i:s') . ". Please try again later.";
}

// Function to auto-disable expired accounts
function autoDisableExpiredAccounts($pdo) {
    try {
        $now = date('Y-m-d H:i:s');
        $stmt = $pdo->prepare("UPDATE users SET is_enabled = 0, enabled_until = NULL WHERE is_enabled = 1 AND enabled_until IS NOT NULL AND enabled_until <= ?");
        $stmt->execute([$now]);
    } catch (Exception $e) {
        // Log error but don't break login process
        error_log("Error auto-disabling expired accounts: " . $e->getMessage());
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $password = $_POST['password'];

    if (empty($username) || empty($password)) {
        $error = "Please fill in all fields.";
    } else {
        // Auto-disable expired accounts before checking login
        autoDisableExpiredAccounts($pdo);

        $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            // Check if account is enabled
            if (!isAccountEnabled($user)) {
                $error = getEnableMessage($user);
            } else {
                // If account was temporarily enabled but now expired, update the database
                if (isset($user['is_enabled']) && $user['is_enabled'] &&
                    $user['enabled_until'] !== null &&
                    new DateTime() >= new DateTime($user['enabled_until'])) {

                    try {
                        $updateStmt = $pdo->prepare("UPDATE users SET is_enabled = 0, enabled_until = NULL WHERE id = ?");
                        $updateStmt->execute([$user['id']]);

                        // Refresh user data
                        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                        $stmt->execute([$user['id']]);
                        $user = $stmt->fetch();
                    } catch (Exception $e) {
                        error_log("Error updating expired account: " . $e->getMessage());
                    }
                }

                $_SESSION['user'] = $user;
                header("Location: dashboard.php");
                exit;
            }
        } else {
            $error = "Invalid username or password.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Login</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" />
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
           background: #020024;
background: linear-gradient(0deg,rgba(2, 0, 36, 1) 0%, rgba(2, 90, 163, 1) 98%);
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
            margin: 0;
        }
        .login-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 2.5rem 3rem;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            width: 380px;
            text-align: center;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        h2 {
            margin-bottom: 1.5rem;
            color: #333;
            font-weight: 600;
        }
        .input-group {
            position: relative;
            margin-bottom: 1.5rem;
        }
        input[type=text], input[type=password] {
            width: 100%;
            padding: 0.875rem 1rem 0.875rem 2.5rem;
            border: 2px solid #e1e5e9;
            border-radius: 8px;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.9);
            box-sizing: border-box;
        }
        input[type=text]:focus, input[type=password]:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
            background: white;
        }
        .input-icon {
            position: absolute;
            left: 0.875rem;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
            font-size: 1rem;
        }
        button {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            border: none;
            padding: 0.875rem 1.5rem;
            font-size: 1rem;
            border-radius: 8px;
            cursor: pointer;
            width: 100%;
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 0.5rem;
            font-weight: 600;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
        }
        button:hover {
            background: linear-gradient(135deg, #000, #003d82);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.4);
        }
        button:active {
            transform: translateY(0);
        }
        .error {
            color: #dc3545;
            margin-bottom: 1.5rem;
            padding: 1rem;
            background: rgba(220, 53, 69, 0.1);
            border: 1px solid rgba(220, 53, 69, 0.2);
            border-radius: 8px;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        .login-header {
            margin-bottom: 2rem;
        }
        .lock-icon {
            font-size: 3rem;
            color: #007bff;
            margin-bottom: 1rem;
        }
        .welcome-text {
            color: #6c757d;
            font-size: 0.9rem;
            margin-bottom: 0;
        }
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }
        .shake {
            animation: shake 0.5s ease-in-out;
        }
    </style>
</head>
<body>
<div class="login-container">
    <div class="login-header">
        <div class="lock-icon">
               <img src="https://kytisa.com/wp-content/uploads/2025/05/logo-kytisa-chat-.png" alt="Kytisa chat" width="auto" height="90">
        </div>
    
        <p class="welcome-text">Connectez-vous pour accéder à votre compte</p>
    </div>

    <?php if (!empty($error)): ?>
        <div class="error" id="error-message">
            <i class="fa-solid fa-circle-exclamation"></i>
            <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <form method="POST" id="loginForm">
        <div class="input-group">
            <i class="fa-solid fa-user input-icon"></i>
            <input type="text" name="username" placeholder="Username" required autofocus autocomplete="username" />
        </div>

        <div class="input-group">
            <i class="fa-solid fa-lock input-icon"></i>
            <input type="password" name="password" placeholder="Password" required autocomplete="current-password" />
        </div>

        <button type="submit" id="loginButton">
            <i class="fa-solid fa-right-to-bracket"></i>
            Sign In
        </button>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('loginForm');
    const button = document.getElementById('loginButton');
    const errorMessage = document.getElementById('error-message');

    // Add shake animation to error message
    if (errorMessage) {
        errorMessage.classList.add('shake');
        setTimeout(() => {
            errorMessage.classList.remove('shake');
        }, 500);
    }

    // Form submission with loading state
    form.addEventListener('submit', function() {
        button.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Signing In...';
        button.disabled = true;
    });

    // Auto-focus on first empty field
    const inputs = form.querySelectorAll('input');
    inputs.forEach(input => {
        if (!input.value) {
            input.focus();
            return false;
        }
    });
});
</script>
</body>
</html>
