document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('search-input');
    const matchCountSpan = document.getElementById('match-count');
    const prevButton = document.getElementById('prev-button');
    const nextButton = document.getElementById('next-button');
    const closeButton = document.getElementById('close-button');
    const searchBar = document.getElementById('search-bar');

    // CRITICAL: Check if DOM elements are found. If not, log an error.
    if (!searchInput || !matchCountSpan || !prevButton || !nextButton || !closeButton || !searchBar) {
        console.error('Custom Search Bar Initialization Error: One or more required DOM elements not found. Please check your HTML IDs in the shortcode output.', {
            searchInput: searchInput,
            matchCountSpan: matchCountSpan,
            prevButton: prevButton,
            nextButton: nextButton,
            closeButton: closeButton,
            searchBar: searchBar
        });
        return; // Stop script execution if elements are missing
    }
    console.log('Custom Search Bar: All required DOM elements found. Script initialized.');

    // --- NEW: Define the selector for the content area to search ---
    // Common WordPress content classes are: .entry-content, .post-content, .page-content
    // Inspect your theme's HTML to find the correct class/ID that wraps the main text.
    const contentSelector = '.entry-content, .post-content, .page-content';
    let contentContainer = document.querySelector(contentSelector);

    // CRITICAL: Check if the content container is found.
    if (!contentContainer) {
        // If content container is not found, fall back to body but warn
        console.warn(`Custom Search Bar: Content container "${contentSelector}" not found. Falling back to searching document.body. Please check your theme's HTML for the correct content wrapper class/ID.`);
        contentContainer = document.body;
    } else {
        console.log('Custom Search Bar: Content will be searched within:', contentContainer);
    }
    // --- END NEW ---

    let allMatches = []; // Stores references to the <mark> elements
    let currentMatchIndex = -1;

    /**
     * Clears all existing highlights from the document.
     */
    function clearHighlights() {
        console.log('clearHighlights: Clearing existing highlights.');
        for (let i = allMatches.length - 1; i >= 0; i--) {
            const markElement = allMatches[i];
            if (markElement && markElement.parentNode) {
                const textNode = document.createTextNode(markElement.textContent);
                markElement.parentNode.replaceChild(textNode, markElement);
            }
        }
        allMatches = [];
        currentMatchIndex = -1;
        updateMatchCount();
        console.log('clearHighlights: Highlights cleared.');
    }

    /**
     * Finds all occurrences of the search text and highlights them.
     * @param {string} searchText - The text to search for.
     */
    function highlightAll(searchText) {
        console.log('highlightAll: Search initiated for:', searchText);
        clearHighlights(); // Always clear previous highlights first

        if (!searchText) {
            console.log('highlightAll: Search text is empty. No highlights.');
            return;
        }

        const regex = new RegExp(searchText, 'gi'); // Global, case-insensitive search

        // NEW: Use contentContainer as the root for TreeWalker
        const walker = document.createTreeWalker(
            contentContainer, // Changed from document.body to contentContainer
            NodeFilter.SHOW_TEXT,
            {
                acceptNode: function(node) {
                    // Reject nodes from script, style, or already highlighted marks
                    if (node.parentNode.nodeName === 'SCRIPT' ||
                        node.parentNode.nodeName === 'STYLE' ||
                        node.parentNode.nodeName === 'MARK') {
                        return NodeFilter.FILTER_REJECT;
                    }

                    // Reject nodes that are part of the search bar itself
                    if (node.parentNode.closest && node.parentNode.closest('#search-bar')) {
                        return NodeFilter.FILTER_REJECT;
                    }

                    // Check if the node's parent element or any ancestor is hidden by CSS
                    let elem = node.parentElement;
                    while (elem && elem !== contentContainer) { // Changed condition to contentContainer
                        const style = window.getComputedStyle(elem);
                        if (style.display === 'none' || style.visibility === 'hidden' || style.opacity === '0') {
                            return NodeFilter.FILTER_REJECT;
                        }
                        elem = elem.parentElement;
                    }
                    return NodeFilter.FILTER_ACCEPT;
                }
            },
            false
        );

        let node;
        const nodesToProcess = [];

        while ((node = walker.nextNode())) {
            nodesToProcess.push(node);
        }
        console.log('highlightAll: Number of searchable text nodes found:', nodesToProcess.length);

        nodesToProcess.forEach(textNode => {
            const parent = textNode.parentNode;
            const text = textNode.nodeValue;
            let match;
            let lastIndex = 0;

            const fragment = document.createDocumentFragment();
            let hasMatchesInNode = false;

            while ((match = regex.exec(text)) !== null) {
                hasMatchesInNode = true;
                if (match.index > lastIndex) {
                    fragment.appendChild(document.createTextNode(text.substring(lastIndex, match.index)));
                }

                const mark = document.createElement('mark');
                mark.className = 'highlighted-match';
                mark.textContent = match[0];
                fragment.appendChild(mark);
                allMatches.push(mark);

                lastIndex = regex.lastIndex;
            }

            if (hasMatchesInNode && lastIndex < text.length) {
                fragment.appendChild(document.createTextNode(text.substring(lastIndex)));
            }

            if (hasMatchesInNode) {
                parent.replaceChild(fragment, textNode);
            }
        });

        console.log('highlightAll: Total matches found:', allMatches.length);
        if (allMatches.length > 0) {
            currentMatchIndex = 0;
            navigateToMatch(0);
        } else {
            currentMatchIndex = -1;
        }
        updateMatchCount();
    }

    /**
     * Navigates to the next or previous match, highlighting it and scrolling into view.
     * @param {number} direction - 1 for next, -1 for previous.
     */
    function navigateToMatch(direction) {
        if (allMatches.length === 0) return;

        console.log('navigateToMatch: Navigating with direction:', direction, 'Current index:', currentMatchIndex);

        if (currentMatchIndex >= 0 && allMatches[currentMatchIndex]) {
            allMatches[currentMatchIndex].classList.remove('current-match');
        }

        currentMatchIndex += direction;

        if (currentMatchIndex >= allMatches.length) {
            currentMatchIndex = 0;
        } else if (currentMatchIndex < 0) {
            currentMatchIndex = allMatches.length - 1;
        }

        const currentMatchElement = allMatches[currentMatchIndex];
        if (currentMatchElement) {
            currentMatchElement.classList.add('current-match');
            currentMatchElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
        updateMatchCount();
        console.log('navigateToMatch: New current index:', currentMatchIndex, 'Element:', currentMatchElement);
    }

    /**
     * Updates the match count display (e.g., "1/5").
     */
    function updateMatchCount() {
        const total = allMatches.length;
        const current = total > 0 ? currentMatchIndex + 1 : 0;
        matchCountSpan.textContent = `${current}/${total}`;
        console.log('updateMatchCount: Count updated to:', matchCountSpan.textContent);
    }

    // --- Event Listeners ---

    // 'input' event: Fires immediately when the value of the input changes (typing, pasting, autofill)
    searchInput.addEventListener('input', (event) => {
        console.log('Event: Input detected. Input value:', event.target.value);
        highlightAll(event.target.value); // Trigger the search and highlight
    });

    // 'keydown' event: For Enter/Shift+Enter navigation
    searchInput.addEventListener('keydown', (event) => {
        if (event.key === 'Enter') {
            event.preventDefault(); // Prevent form submission or new line in input
            if (event.shiftKey) {
                navigateToMatch(-1); // Shift + Enter: Previous match
            } else {
                navigateToMatch(1); // Enter: Next match
            }
            console.log('Event: Keydown - Enter pressed. Shift key:', event.shiftKey);
        }
    });

    nextButton.addEventListener('click', () => {
        console.log('Event: Next button clicked.');
        navigateToMatch(1);
    });

    prevButton.addEventListener('click', () => {
        console.log('Event: Previous button clicked.');
        navigateToMatch(-1);
    });

    closeButton.addEventListener('click', () => {
        console.log('Event: Close button clicked.');
        clearHighlights();
        searchInput.value = ''; // Clear the input field
        searchInput.focus(); // Keep focus on the input after closing
    });

    // Initial setup: Display "0/0" when the page loads
    updateMatchCount();
});